<?php
/**
 * Woffice - Latest Posts Widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Latest_Posts_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Latest Posts widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-latest-posts';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Latest Posts widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Latest Posts', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Latest Posts widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-recent-posts';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Latest Posts widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Latest Posts content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

        $taxonomy = $this->get_latest_post_taxonomy();

		$this->start_controls_section(
			'woffice_latest_posts_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_latest_posts_title_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_latest_posts_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Latest Posts', 'woffice' ),
				'placeholder' => esc_html__( 'Latest Posts', 'woffice' ),
			]
		);

        $this->add_control(
			'number_of_posts',
			[
				'label' => esc_html__( 'Number of Posts', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 6,
			]
		);

        if(!empty($taxonomy)){
            foreach ( $taxonomy as $index => $tax ) {
                $terms = get_terms( $index );

                $related_tax = array();

                if ( ! empty( $terms ) ) {
                    foreach ( $terms as $t_index => $t_obj ) {
                        $related_tax[ $t_obj->slug ] = $t_obj->name;
                    }
                    // Add control for all taxonomies.
                    $this->add_control(
                        'tax_' . $index . '_filter',
                        array(
                            /* translators: %s label */
                            'label'       => sprintf( __( '%s Filter', 'uael' ), $tax->label ),
                            'type'        => Controls_Manager::SELECT2,
                            'multiple'    => true,
                            'default'     => '',
                            'label_block' => true,
                            'options'     => $related_tax,
                        )
                    );
                }
            }
        }


		$this->end_controls_section();

	}

    public function get_latest_post_taxonomy(){

        $taxonomies = get_object_taxonomies( 'post', 'objects' );
		$data       = array();

		foreach ( $taxonomies as $tax_slug => $tax ) {
			if ( ! $tax->public || ! $tax->show_ui ) {
				continue;
			}

			$data[ $tax_slug ] = $tax;
		}

		return apply_filters( 'woffice_post_loop_taxonomies', $data, $taxonomies, 'post' );
    }

    public function build_woffice_latest_posts_query($settings) {

        $posts_per_page = $settings['number_of_posts'];

        $query_args = array(
            'post_type' => 'post',
            'posts_per_page' => $posts_per_page,
            'post_status' => array( 'publish'),
        );

        $category = $settings['tax_category_filter'];

        if (!empty($category) && $category != "all") {

            $the_tax = array(array(
                'taxonomy' => 'category',
                'terms' => $category,
                'field' => 'slug',
            ));

            $query_args['tax_query'] = $the_tax;
        }

        // GET WIKI POSTS
        $widget_posts_query = new WP_Query($query_args);

        return $widget_posts_query;
    }

    /**
	 * Render output from wiki on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_latest_posts_title'];
        $widget_posts_query = $this->build_woffice_latest_posts_query($settings);
        ?>
            <div class="widget widget_woffice_recent_posts">
                <div class="box">
                    <div class="widget_recent_posts intern-box box-title">
                        <?php
                            echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_latest_posts_title_tag'] ), $this->get_render_attribute_string( 'woffice_latest_posts_title' ), $title );
                        ?>
                    </div>
                    <!-- WIDGET -->
	                <ul class="list-styled list-recent-posts pl-0">
                        <?php 
							if(!empty($widget_posts_query)){
								foreach( $widget_posts_query->posts as $post_item ) : ?>
									<li class="list-recent-posts">
										<div class="row recent-posts-item align-items-center">
											<div  class="col-md-4 recent-posts-thumb">
												<?php echo get_the_post_thumbnail($post_item->ID, 'thumbnail'); ?>
											</div>  
											<div class="col-md-8 recent-posts-content pl-0">
												<h3 class="recent-posts-title mb-0"><?php echo $post_item->post_title; ?></h3>
												<span class="font-rg1"><?php echo get_the_date('F j, Y g:i a',$post_item->ID); ?></span>
											</div>
										</div>
									</li>
							<?php endforeach; }?>
                    </ul>
                </div>    
            </div>
		<?php

	}
}