<?php
/**
 * MU Plugin: AH Map folder-based uploads into multiple ACF repeaters (one repeater per doc type).
 *
 * Folder pattern (under /uploads):
 *   "NOM PRENOM type"
 * Example:
 *   "ARNAUD ALEXANDRE rib" -> files inside go into repeater mapped for type "rib"
 *
 * Each file becomes one repeater row containing:
 *   - the attachment ID in a sub field (File field)
 *
 * Run (admin only):
 *   https://ton-site.tld/?ah_map_multi_repeaters=1
 */

if (!defined('ABSPATH')) exit;

add_action('init', function () {
    if (!is_user_logged_in() || !current_user_can('administrator')) return;
    if (!isset($_GET['ah_map_multi_repeaters']) || $_GET['ah_map_multi_repeaters'] !== '1') return;

    @set_time_limit(0);
    @ini_set('max_execution_time', '0');
    @ini_set('memory_limit', '512M');
    if (function_exists('session_write_close')) @session_write_close();

    if (!function_exists('update_field') || !function_exists('get_field')) {
        wp_die("ACF n'est pas actif (get_field/update_field introuvables).");
    }

    // =======================
    // CONFIG À REMPLIR
    // =======================
    $post_type = 'clients'; // ton CPT

    /**
     * Associe le "type" (dernier mot du dossier) -> repeater + sous-champ fichier
     *
     * Remplis avec tes 7 types et les noms ACF (field name) exacts.
     * Exemple:
     * 'rib' => ['repeater' => 'rep_rib', 'file_subfield' => 'fichier'],
     */
    $type_map = [
        'cni'        => ['repeater' => 'cni',        'file_subfield' => 'cni_fichiers'],
        'rib'        => ['repeater' => 'rib',        'file_subfield' => 'rib_fichiers'],
        'der'        => ['repeater' => 'der',        'file_subfield' => 'der_fichiers'],
        'justif'     => ['repeater' => 'justif',     'file_subfield' => 'justif_fichiers'],
        'lettre'     => ['repeater' => 'lettre',     'file_subfield' => 'lettre_fichiers'],
        'situation'  => ['repeater' => 'situation',  'file_subfield' => 'situation_fichiers'],
        'facultatif' => ['repeater' => 'facultatif', 'file_subfield' => 'facultatif_fichiers'],
    ];

    /**
     * Comportement si le repeater contient déjà des lignes :
     * - 'append'  : ajoute à la fin
     * - 'replace' : remplace entièrement par ce qu'on trouve dans les dossiers
     */
    $mode = 'append';

    /**
     * Déduplication :
     * - true  : n'ajoute pas un attachment ID déjà présent dans le repeater
     * - false : ajoute sans vérifier (risque de doublons si relance)
     */
    $dedupe = true;
    // =======================

    $upload = wp_upload_dir();
    $basedir = $upload['basedir'];

    $dirs = glob($basedir . '/*', GLOB_ONLYDIR);
    if (!$dirs) wp_die("Aucun dossier trouvé dans uploads.");

    $updated_clients = 0;
    $skipped_clients = 0;
    $added_rows = 0;
    $skipped_rows = 0;

    foreach ($dirs as $dirPath) {
        $folder = basename($dirPath);

        // Split sur le dernier espace : fullname + type
        $pos = strrpos($folder, ' ');
        if ($pos === false) continue;

        $fullname = trim(substr($folder, 0, $pos)); // ex "ARNAUD ALEXANDRE"
        $type     = strtolower(trim(substr($folder, $pos + 1))); // ex "rib"

        if ($fullname === '' || $type === '') continue;
        if (!isset($type_map[$type])) continue;

        $repeater_field = $type_map[$type]['repeater'];
        $file_subfield  = $type_map[$type]['file_subfield'];

        // Trouver la fiche client par titre
        $existing = get_page_by_title($fullname, OBJECT, $post_type);
        if (!$existing) { $skipped_clients++; continue; }
        $client_id = (int) $existing->ID;

        // Récupérer tous les attachments dont le chemin est dans ce dossier
        $attachments = get_posts([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'fields'         => 'ids',
            'posts_per_page' => -1,
            'meta_query'     => [[
                'key'     => '_wp_attached_file',
                'value'   => $folder . '/',
                'compare' => 'LIKE',
            ]],
            'orderby' => 'ID',
            'order'   => 'ASC',
        ]);

        if (empty($attachments)) continue;

        // Charger repeater existant si append
        $rows = [];
        $existing_ids = [];

        if ($mode === 'append') {
            $current = get_field($repeater_field, $client_id);
            if (is_array($current)) {
                $rows = $current;

                if ($dedupe) {
                    foreach ($rows as $r) {
                        if (isset($r[$file_subfield]) && is_numeric($r[$file_subfield])) {
                            $existing_ids[(int)$r[$file_subfield]] = true;
                        } elseif (isset($r[$file_subfield]['ID'])) {
                            $existing_ids[(int)$r[$file_subfield]['ID']] = true;
                        }
                    }
                }
            }
        }

        $added_this_client = 0;

        foreach ($attachments as $att_id) {
            $att_id = (int) $att_id;

            if ($dedupe && isset($existing_ids[$att_id])) {
                $skipped_rows++;
                continue;
            }

            $rows[] = [
                $file_subfield => $att_id,
            ];

            $added_rows++;
            $added_this_client++;
            if ($dedupe) $existing_ids[$att_id] = true;
        }

        // Écrire le repeater seulement si on a ajouté ou si replace
        if ($mode === 'replace' || $added_this_client > 0) {
            update_field($repeater_field, $rows, $client_id);
            $updated_clients++;
        }
    }

    wp_die(
        "Mapping multi-repeaters terminé.\n" .
        "Clients mis à jour: {$updated_clients}\n" .
        "Clients ignorés (pas trouvés): {$skipped_clients}\n" .
        "Lignes ajoutées: {$added_rows}\n" .
        "Lignes ignorées (déjà présentes): {$skipped_rows}\n" .
        "Mode: {$mode}\n" .
        "Déduplication: " . ($dedupe ? "ON" : "OFF")
    );
});
