<?php
/**
 * Plugin Name: AH - Client CSV Importer (ACF)
 * Description: Import clients from a CSV file into a CPT and map columns to ACF fields. Unique key = post title (prenom + nom).
 * Version: 1.1.0
 * Author: Arholding
 */

if (!defined('ABSPATH')) exit;

class AH_Client_CSV_Importer {
    const MENU_SLUG = 'ah-client-csv-importer';

    public function __construct() {
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('admin_post_ah_client_csv_import', [$this, 'handle_import']);
    }

    public function register_menu() {
        add_management_page(
            'Import clients (CSV)',
            'Import clients (CSV)',
            'manage_options',
            self::MENU_SLUG,
            [$this, 'render_page']
        );
    }

    private function default_config() {
        return [
            'post_type'   => 'clients',
            'post_status' => 'publish',

            /**
             * ✅ Clé unique = titre du post (prenom + nom)
             * On cherche un post existant via son titre exact.
             */
            'unique_mode' => 'post_title',

            /**
             * Colonnes CSV utilisées pour construire le titre et la clé unique.
             * (Chez toi : nom_client_ag + prenom_client_ag)
             */
'lastname_csv_column'  => 'xnom',
'firstname_csv_column' => 'xprenom',



            /**
             * Délimiteur CSV (auto si vide)
             */
            'delimiter' => '',

            /**
             * Mapping CSV -> champs ACF (V2)
             * Clé = colonne CSV, Valeur = field_name ACF
             */
            'field_map' => [
                'lieu_de_naissance' => 'lieu_de_naissance',
                'pays_de_naissance' => 'pays_de_naissance',
                'type' => 'type',
                'date_de_deces' => 'date_de_deces',
                'titre' => 'titre',
                'situation_familiale' => 'situation_familiale',
                'nationalite' => 'nationalite',
                'pays_de_residence_fiscale' => 'pays_de_residence_fiscale',
                'nationalite_copier' => 'nationalite_copier',
                'capacite_juridique' => 'capacite_juridique',
                'profession' => 'profession',
                'revenus_annuels' => 'revenus_annuels',
                'patrimoine_global' => 'patrimoine_global',
                'origine' => 'origine',
                'xrue_perso' => 'xrue_perso',
                'rue_pro' => 'rue_pro',
                'xcode_postal_perso' => 'xcode_postal_perso',
                'code_postal_pro' => 'code_postal_pro',
                'xville_perso' => 'xville_perso',
                'ville_pro' => 'ville_pro',
                'pays_perso' => 'pays_perso',
                'pays_pro' => 'pays_pro',
                'le_client_est' => 'le_client_est',
                'rue_autre' => 'rue_autre',
                'code_postal_autre' => 'code_postal_autre',
                'ville_autre' => 'ville_autre',
                'pays_autre' => 'pays_autre',
                'mobile' => 'mobile',
                'emailperso' => 'emailperso',
                'domicile' => 'domicile',
                'emailpro' => 'emailpro',
                'domicile_2' => 'domicile_2',
                'emailautre' => 'emailautre',
                'bureau' => 'bureau',
                'bureau2' => 'bureau2',
                'notes' => 'notes',
                'xnom_de_naissance' => 'xnom_de_naissance',
                'xdatenaissance' => 'date_naissance_client_ag',
                'xnom' => 'nom_client_ag',
                'xprenom' => 'prenom_client_ag',
            ],

            'ignore_columns' => [],
        ];
    }

    public function render_page() {
        if (!current_user_can('manage_options')) return;

        $cfg = $this->default_config();
        ?>
        <div class="wrap">
            <h1>Import clients (CSV)</h1>

            <p>
                CPT cible : <code><?php echo esc_html($cfg['post_type']); ?></code><br>
                Clé unique : <strong>titre</strong> = <code><?php echo esc_html($cfg['firstname_csv_column']); ?></code> + espace + <code><?php echo esc_html($cfg['lastname_csv_column']); ?></code>
            </p>

            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
                <?php wp_nonce_field('ah_client_csv_import'); ?>
                <input type="hidden" name="action" value="ah_client_csv_import" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="csv_file">Fichier CSV</label></th>
                        <td><input type="file" name="csv_file" id="csv_file" accept=".csv,text/csv" required /></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="dry_run">Mode test (dry run)</label></th>
                        <td>
                            <label>
                                <input type="checkbox" name="dry_run" id="dry_run" value="1" checked />
                                Ne rien écrire, juste analyser et afficher un rapport.
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="max_rows">Limiter le nombre de lignes</label></th>
                        <td>
                            <input type="number" name="max_rows" id="max_rows" value="0" min="0" />
                            <p class="description">0 = pas de limite.</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button('Importer'); ?>
            </form>

            <?php if (isset($_GET['ah_report']) && $_GET['ah_report'] === '1' && isset($_GET['msg'])): ?>
                <hr />
                <h2>Rapport</h2>
                <pre style="background:#fff; border:1px solid #ccd0d4; padding:12px; max-height: 520px; overflow:auto;"><?php
                    echo esc_html(base64_decode(sanitize_text_field($_GET['msg'])));
                ?></pre>
            <?php endif; ?>
        </div>
        <?php
    }

    public function handle_import() {
        // Tenter d'éviter les timeouts côté PHP / nginx/apache
@set_time_limit(0);
@ini_set('max_execution_time', '0');
@ini_set('memory_limit', '512M');

// Évite que la session PHP bloque / expire
if (function_exists('session_write_close')) {
    @session_write_close();
}

// Si FastCGI (souvent nginx), demande au serveur de ne pas couper la réponse
if (function_exists('fastcgi_finish_request')) {
    // Ne pas l'appeler ici : on veut garder la réponse pour afficher le rapport.
}

        if (!current_user_can('manage_options')) wp_die('Permissions insuffisantes.');
        check_admin_referer('ah_client_csv_import');

        $cfg = $this->default_config();
        $dry_run  = !empty($_POST['dry_run']);
        $max_rows = isset($_POST['max_rows']) ? max(0, intval($_POST['max_rows'])) : 0;

        if (empty($_FILES['csv_file']['tmp_name'])) {
            $this->redirect_with_report("Aucun fichier uploadé.");
        }

        $tmp = $_FILES['csv_file']['tmp_name'];
        $report = [];

        $raw = file_get_contents($tmp);
        if ($raw === false) $this->redirect_with_report("Impossible de lire le fichier.");

        // Strip UTF-8 BOM
        $raw = preg_replace('/^\xEF\xBB\xBF/', '', $raw);

        // Try convert to UTF-8 if needed (Excel)
        if (!mb_check_encoding($raw, 'UTF-8')) {
            $raw = mb_convert_encoding($raw, 'UTF-8', 'ISO-8859-1');
        }

        $lines = preg_split("/\r\n|\n|\r/", $raw);
        $lines = array_values(array_filter($lines, fn($l) => trim($l) !== ''));

        if (count($lines) < 2) $this->redirect_with_report("CSV vide ou sans données.");

        // Detect delimiter
        $delimiter = $cfg['delimiter'];
        if ($delimiter === '') {
            $sample = $lines[0];
            $semi  = substr_count($sample, ';');
            $comma = substr_count($sample, ',');
            $delimiter = ($semi >= $comma) ? ';' : ',';
        }

        $header = str_getcsv($lines[0], $delimiter);
        $header = array_map([$this, 'clean_header'], $header);

        $report[] = "Delimiter: {$delimiter}";
        $report[] = "Colonnes: " . implode(' | ', $header);

        $idx = array_flip($header);

        $processed = $created = $updated = $skipped = $errors = 0;
        
        $last_col  = $this->clean_header($cfg['lastname_csv_column']);
        $first_col = $this->clean_header($cfg['firstname_csv_column']);


        if (!isset($idx[$first_col]) || !isset($idx[$last_col])) {
            $this->redirect_with_report(
                "❌ Colonnes nécessaires absentes.\n" .
                "Attendu: {$cfg['firstname_csv_column']} + {$cfg['lastname_csv_column']}\n" .
                "Trouvé: " . implode(' | ', $header)
            );
        }

        for ($i = 1; $i < count($lines); $i++) {
            if ($max_rows > 0 && $processed >= $max_rows) break;

            $row = str_getcsv($lines[$i], $delimiter);
            if (count($row) < count($header)) $row = array_pad($row, count($header), '');

            $data = [];
            foreach ($header as $col) {
                $data[$col] = isset($idx[$col]) ? trim((string)$row[$idx[$col]]) : '';
            }

            $non_empty = array_filter($data, fn($v) => $v !== '');
            if (empty($non_empty)) { $skipped++; continue; }

            $processed++;

            $post_title = $this->build_title_from_first_last($data, $first_col, $last_col);
            if ($post_title === '') { $skipped++; continue; }

            try {
                // Find existing by title
                $existing = get_page_by_title($post_title, OBJECT, $cfg['post_type']);
                $post_id = $existing ? (int)$existing->ID : 0;

                if (!$post_id) {
                    if ($dry_run) {
                        $created++;
                        $report[] = "DRY RUN: créer « {$post_title} »";
                        continue;
                    }

$post_id = wp_insert_post([
    'post_type'   => $cfg['post_type'],
    'post_status' => $cfg['post_status'],
    'post_title'  => $post_title,
], true);

if (is_wp_error($post_id) || !$post_id) {
    // gérer l'erreur...
} else {
    // Catégorie "Clients" (term_id = 5)
    wp_set_post_terms($post_id, [5], 'category', false);
}


                    if (is_wp_error($post_id)) {
                        $errors++;
                        $report[] = "❌ Ligne {$i}: erreur création « {$post_title} »: " . $post_id->get_error_message();
                        continue;
                    }
                    $created++;
                } else {
                    if (!$dry_run) {
                        wp_update_post(['ID' => $post_id, 'post_title' => $post_title]);
                    }
                    $updated++;
                }

                // Apply mapping
                foreach ($cfg['field_map'] as $csv_col_raw => $acf_field) {
                    $csv_col = $this->clean_header($csv_col_raw);
                    if (in_array($csv_col, $cfg['ignore_columns'], true)) continue;

                    $val = $data[$csv_col] ?? '';

                    // Ne pas écraser avec du vide (ré-import safe)
                    if ($val === '') continue;

                    if ($dry_run) continue;

                    if (function_exists('update_field')) {
                        update_field($acf_field, $val, $post_id);
                    } else {
                        update_post_meta($post_id, $acf_field, $val);
                    }
                }

            } catch (\Throwable $e) {
                $errors++;
                $report[] = "❌ Ligne {$i}: exception: " . $e->getMessage();
            }
        }

        $report[] = "----";
        $report[] = "Terminé.";
        $report[] = "Lignes traitées: {$processed}";
        $report[] = "Créés: {$created}";
        $report[] = "Mises à jour: {$updated}";
        $report[] = "Ignorés: {$skipped}";
        $report[] = "Erreurs: {$errors}";
        $report[] = "Mode: " . ($dry_run ? "DRY RUN" : "ÉCRITURE");

        $this->redirect_with_report(implode("\n", $report));
    }

    private function clean_header($s) {
        $s = trim((string)$s);
        $s = preg_replace('/\s+/', ' ', $s);
        return $s;
    }

    private function normalize_title($s) {
        $s = trim((string)$s);
        // Réduire espaces multiples
        $s = preg_replace('/\s+/', ' ', $s);
        return $s;
    }

    private function build_title_from_first_last(array $data, $first_col, $last_col) {
        $prenom = $data[$first_col] ?? '';
        $nom    = $data[$last_col] ?? '';

        $title = $this->normalize_title(trim($nom . ' ' . $prenom));

        return $title;
    }

    private function redirect_with_report($report_text) {
        $msg = base64_encode($report_text);
        $url = add_query_arg([
            'page' => self::MENU_SLUG,
            'ah_report' => '1',
            'msg' => rawurlencode($msg),
        ], admin_url('tools.php'));
        wp_safe_redirect($url);
        exit;
    }
}

new AH_Client_CSV_Importer();
