<?php
/**
 * Totaux clients / contrats / conformité (global + par user)
 * + Totaux prospects (global + par user)
 * FIX: exclure des TOTAUX globaux les clients dont conseillers_manual contient:
 *  - non_assigné
 *  - sans_effet
 * (mais NE PAS exclure des stats par user)
 */


/* ------------------------------------------------------
   🔵 Helper : détecte "Non assigné" / "Sans effet"
   ------------------------------------------------------ */
function ah_is_special_conseiller_client($client_id): bool {
    $saved = get_post_meta($client_id, 'conseillers_manual', true);
    if (!is_array($saved)) return false;
    return in_array('non_assigné', $saved, true) || in_array('sans_effet', $saved, true);
}


// ------------------------------------------------------
// 🔵 Fonction : total contrats agrégation + hors-agrégation
// ------------------------------------------------------
function get_total_contrats_client($post_id) {
    $prenom_client = trim(get_field('prenom_client_ag', $post_id));
    $nom_client    = trim(get_field('nom_client_ag', $post_id));
    if (!$prenom_client || !$nom_client) return 0;

    // ===== Contrats via agrégation =====
    $args_agreg = [
        'post_type'      => ['contrats','contrats-avec-ucs'],
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'fields'         => 'ids',
        'meta_query'     => [
            'relation' => 'AND',
            ['key'=>'nom_client_contrat','value'=>$nom_client,'compare'=>'='],
            ['key'=>'prenom_client_contrat','value'=>$prenom_client,'compare'=>'=']
        ]
    ];
    $query_agreg = new WP_Query($args_agreg);
    $nb_agreg = $query_agreg->found_posts;

    // ===== Contrats hors agrégation =====
    $args_hag = [
        'post_type'      => 'contrats_hag',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'fields'         => 'ids',
        'meta_query'     => [
            ['key'=>'client_horsag','value'=>$prenom_client.' '.$nom_client,'compare'=>'=']
        ]
    ];
    $query_hag = new WP_Query($args_hag);
    $nb_hag = $query_hag->found_posts;

    return $nb_agreg + $nb_hag;
}



add_action('save_post', function($post_id) {

    $post_type = get_post_type($post_id);

    if (!in_array($post_type, ['clients', 'contrats', 'contrats-avec-ucs', 'contrats_hag'])) {
        return;
    }

    // (info only)
    $current_user_id = get_current_user_id();

    // ---- Tous les clients ----
    $clients = get_posts([
        'post_type' => 'clients',
        'posts_per_page' => -1,
        'fields' => 'ids',
    ]);

    $total_clients = 0;       // global (hors non_assigné / sans_effet)
    $total_contrats = 0;      // global (hors non_assigné / sans_effet)
    $somme_conformite = 0;    // somme % pour moyenne (hors non_assigné / sans_effet)
    $clients_par_user = [];   // stats par conseiller (inclut non_assigné / sans_effet)

    foreach ($clients as $client_id) {

        /* ------------------------------------------------------
           1) Vérifie "Sans effet" (archivage)
           ------------------------------------------------------ */
        $archiver_values = get_field('field_691ece664fd59', $client_id);
        $is_actif = true;
        if ($archiver_values && is_array($archiver_values)) {
            $normalized = array_map(function($v){
                $v = strtolower($v);
                $v = str_replace('_',' ', $v);
                $v = iconv('UTF-8','ASCII//TRANSLIT', $v);
                return trim($v);
            }, $archiver_values);
            if (in_array('sans effet', $normalized, true)) $is_actif = false;
        }
        if (!$is_actif) continue;

        /* ------------------------------------------------------
           2) Ne garde que type_de_contact = "Client"
           ------------------------------------------------------ */
        $type = get_field('type_de_contact', $client_id);
        if (empty($type) || !in_array('Client', (array)$type, true)) {
            continue;
        }

        /* ------------------------------------------------------
           3) Conformité (calculé pour global + par user)
           ------------------------------------------------------ */
        $repeater_map = [
            'der'       => ['sub' => 'der_fichiers'],
            'cni'       => ['sub' => 'cni_fichiers'],
            'rib'       => ['sub' => 'rib_fichiers'],
            'justif'    => ['sub' => 'justif_fichiers'],
            'lettre'    => ['sub' => 'lettre_fichiers'],
            'situation' => ['sub' => 'situation_fichiers'],
        ];
        $total_rep = count($repeater_map);
        $found = 0;
        foreach ($repeater_map as $key => $data) {
            $rows = get_field($key, $client_id);
            if (is_array($rows) && !empty($rows)) {
                foreach ($rows as $row) {
                    if (!empty($row[$data['sub']])) { $found++; break; }
                }
            }
        }
        $pourcentage = ($total_rep>0) ? round(($found/$total_rep)*100) : 0;

        /* ------------------------------------------------------
           4) FIX: exclure des TOTAUX globaux les "Non assigné" / "Sans effet"
              (mais pas des stats par user)
           ------------------------------------------------------ */
        $is_special = ah_is_special_conseiller_client($client_id);

        // ---- Totaux globaux (sans non_assigné / sans_effet) ----
        if (!$is_special) {
            $total_clients++;
            $total_contrats += get_total_contrats_client($client_id);
            $somme_conformite += $pourcentage;
        }

        // ---- Par conseiller (ON GARDE y compris non_assigné / sans_effet) ----
        $saved_conseillers = get_post_meta($client_id,'conseillers_manual',true);
        if (is_array($saved_conseillers)) {
            foreach ($saved_conseillers as $uid) {
                if (!isset($clients_par_user[$uid])) {
                    $clients_par_user[$uid] = ['clients'=>0,'contrats'=>0,'conformite'=>0,'nb'=>0];
                }

                $clients_par_user[$uid]['clients']++;
                $clients_par_user[$uid]['contrats'] += get_total_contrats_client($client_id);
                $clients_par_user[$uid]['conformite'] += $pourcentage;
                $clients_par_user[$uid]['nb']++;
            }
        }
    }

    // ---- Stockage global ----
    update_option('total_clients_non_archives', $total_clients);
    update_option('total_contrats_actifs', $total_contrats);
    $moyenne = ($total_clients>0)? round($somme_conformite/$total_clients) : 0;
    update_option('conformite_moyenne', $moyenne);

    // ---- Stockage par utilisateur ----
    foreach ($clients_par_user as $uid => $data) {
        update_user_meta($uid, '_clients_non_archives_user', $data['clients']);
        update_user_meta($uid, '_compte_contrats_actifs_user', $data['contrats']);
        $m = ($data['nb']>0)? round($data['conformite']/$data['nb']):0;
        update_user_meta($uid, '_conformite_totale_user', $m);
    }


    /* =====================================================
       🔵 CALCUL PROSPECTS
       FIX: exclure des TOTAUX globaux les "Non assigné" / "Sans effet"
       (mais pas des totaux par user)
       ===================================================== */
    $prospects_total = 0;     // global (hors non_assigné / sans_effet)
    $prospects_par_user = []; // par user (inclut)

    foreach ($clients as $client_id) {

        // Vérifie "Sans effet" (archivage)
        $archiver_values = get_field('field_691ece664fd59', $client_id);
        $is_actif = true;
        if ($archiver_values && is_array($archiver_values)) {
            $normalized = array_map(function($v){
                $v = strtolower($v);
                $v = str_replace('_',' ', $v);
                $v = iconv('UTF-8','ASCII//TRANSLIT', $v);
                return trim($v);
            }, $archiver_values);
            if (in_array('sans effet', $normalized, true)) $is_actif = false;
        }
        if (!$is_actif) continue;

        // 🔵 NE PREND QUE LES Prospects
        $type = get_field('type_de_contact', $client_id);
        if (empty($type) || !in_array('Prospect', (array)$type, true)) {
            continue;
        }

        // FIX: détecte spécial (non_assigné / sans_effet)
        $is_special = ah_is_special_conseiller_client($client_id);

        // ---- Total global prospects (sans non_assigné / sans_effet) ----
        if (!$is_special) {
            $prospects_total++;
        }

        // ---- Comptage par conseiller (ON GARDE) ----
        $saved_conseillers = get_post_meta($client_id,'conseillers_manual',true);
        if (is_array($saved_conseillers)) {
            foreach ($saved_conseillers as $uid) {
                if (!isset($prospects_par_user[$uid])) {
                    $prospects_par_user[$uid] = ['nb'=>0];
                }
                $prospects_par_user[$uid]['nb']++;
            }
        }
    }

    // ---- Stockage global ----
    update_option('total_prospects_non_archives', $prospects_total);

    // ---- Stockage par utilisateur ----
    foreach ($prospects_par_user as $uid => $data) {
        update_user_meta($uid, '_prospects_non_archives_user', $data['nb']);
    }

}, 20, 1);




/* ------------------------------------------------------
   Shortcodes
   ------------------------------------------------------ */

// Clients (global)
add_shortcode('clients_non_archives', function() {
    $count = get_option('total_clients_non_archives',0);
    return '<div style="text-align:center; font-size:40px; font-weight:bold; color:#fff;">'.$count.'</div>';
});

// Contrats (global)
add_shortcode('compte_contrats_actifs', function() {
    $count = get_option('total_contrats_actifs',0);
    return "<div style='text-align:center; font-size:40px; font-weight:bold; color:#fff;'>{$count}</div>";
});

// Conformité (global)
add_shortcode('conformite_totale', function() {
    $m = get_option('conformite_moyenne',0);
    return "<div style='text-align:center; font-size:40px; font-weight:bold; color:#fff;'>{$m}%</div>";
});

// Clients (user)
add_shortcode('clients_non_archives_user', function() {
    if(!is_user_logged_in()) return '0';
    $m = get_user_meta(get_current_user_id(),'_clients_non_archives_user',true);
    return '<div style="text-align:center; font-size:40px; font-weight:bold; color:#fff;">'.($m?:0).'</div>';
});

// Contrats (user)
add_shortcode('compte_contrats_actifs_user', function() {
    if(!is_user_logged_in()) return '0';
    $m = get_user_meta(get_current_user_id(),'_compte_contrats_actifs_user',true);
    return "<div style='text-align:center; font-size:40px; font-weight:bold; color:#fff;'>".($m?:0)."</div>";
});

// Conformité (user)
add_shortcode('conformite_totale_user', function() {
    if(!is_user_logged_in()) return '0%';
    $m = get_user_meta(get_current_user_id(),'_conformite_totale_user',true);
    return "<div style='text-align:center; font-size:40px; font-weight:bold; color:#fff;'>".($m?:0)."%</div>";
});

// Prospects (global)
add_shortcode('prospects_non_archives', function() {
    $count = get_option('total_prospects_non_archives', 0);
    return '<div style="text-align:center; font-size:40px; font-weight:bold; color:#fff;">'.$count.'</div>';
});

// Prospects (user)
add_shortcode('prospects_non_archives_user', function() {
    if (!is_user_logged_in()) return '0';
    $count = get_user_meta(get_current_user_id(), '_prospects_non_archives_user', true);
    return '<div style="text-align:center; font-size:40px; font-weight:bold; color:#fff;">'.($count ?: 0).'</div>';
});
