<?php

function acf_total_evaluation($filter_user = true) {

    if ($filter_user && !is_user_logged_in()) return 0.0;

    $user_id = $filter_user ? get_current_user_id() : null;
    $total_evaluation = 0.0;

    // Fonction pour parser les montants
    $parse_number = function($raw) {
        if ($raw === null || $raw === '') return 0.0;
        $s = trim((string)$raw);
        $s = str_replace(["\xc2\xa0",'%', '€'], [' ','',''], $s);
        $s = str_replace(',', '.', $s);
        $s = preg_replace('/[^\d\.\-]/u','',$s);
        return ($s === '' ? 0.0 : floatval($s));
    };

    // Récupérer tous les clients
    $clients = get_posts([
        'post_type'      => 'clients',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'fields'         => 'ids',
    ]);

    if (!$clients) return 0.0;

    foreach ($clients as $client_id) {

        // Filtrage par conseiller si nécessaire
        if ($filter_user) {
            $assigned = get_post_meta($client_id, 'conseillers_manual', true);
            if (!is_array($assigned)) $assigned = [];
            if (!in_array($user_id, $assigned)) continue;
        }

        // Récupérer les infos du client
        $prenom = get_field('prenom_client_ag', $client_id);
        $nom    = get_field('nom_client_ag', $client_id);

        // Récupérer tous les contrats liés à ce client
        $args = [
            'post_type'      => ['contrats','contrats-avec-ucs'],
            'posts_per_page' => -1,
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'meta_query'     => [
                'relation' => 'AND',
                [
                    'key'     => 'nom_client_contrat',
                    'value'   => $nom,
                    'compare' => '=',
                ],
                [
                    'key'     => 'prenom_client_contrat',
                    'value'   => $prenom,
                    'compare' => '=',
                ],
            ],
        ];

        $contracts = get_posts($args);

        foreach ($contracts as $contract_id) {
            $evaluation = get_post_meta($contract_id, 'evaluation_contrat', true);
            $total_evaluation += $parse_number($evaluation);
        }
    }

    return $total_evaluation;
}


/* ============================================================
   SHORTCODE 1 : Total évaluation conseiller connecté
   ============================================================ */
add_shortcode('total_evaluation_conseiller', function() {

    $total = acf_total_evaluation(true);
    $formatted_total = number_format($total, 2, ',', ' ') . ' €';

    return '<div style="text-align:center; font-size:32px; font-weight:bold; color:#fff;">' . esc_html($formatted_total) . '</div>';
});


/* ============================================================
   SHORTCODE 2 : Total évaluation tous les clients
   ============================================================ */
add_shortcode('total_evaluation_all', function() {

    $total = acf_total_evaluation(false);
    $formatted_total = number_format($total, 2, ',', ' ') . ' €';

    return '<div style="text-align:center; font-size:32px; font-weight:bold; color:#fff;">' . esc_html($formatted_total) . '</div>';
});













/* ------------------------------------------------------
   🔵 Helper : détecte "Non assigné" / "Sans effet"
   ------------------------------------------------------ */
function ah_is_special_conseiller_client2($client_id): bool {
    $saved = get_post_meta($client_id, 'conseillers_manual', true);
    if (!is_array($saved)) return false;
    return in_array('non_assigné', $saved, true) || in_array('sans_effet', $saved, true);
}


function acf_count_clients_conformes($filter_user = true) {

    // Conseiller connecté si filtrage actif
    $user_id = get_current_user_id();

    // Récupère les clients (IDs uniquement)
    $clients = get_posts([
        'post_type'      => 'clients',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'fields'         => 'ids',
    ]);

    if (!$clients) return [0, 0];

    $total_clients     = 0;
    $clients_conformes = 0;

    // Liste des repeaters
    $repeater_map = [
        'der'       => 'der_fichiers',
        'cni'       => 'cni_fichiers',
        'rib'       => 'rib_fichiers',
        'justif'    => 'justif_fichiers',
        'lettre'    => 'lettre_fichiers',
        'situation' => 'situation_fichiers',
    ];
    $total_rep = count($repeater_map);

    foreach ($clients as $client_id) {

        // ========== FILTRE CONSEILLER ==========
        if ($filter_user === true) {

            $saved_conseillers = get_post_meta($client_id, 'conseillers_manual', true);

            if (empty($saved_conseillers) || !is_array($saved_conseillers)) {
                continue;
            }

            if (!in_array($user_id, $saved_conseillers, false)) {
                // false au lieu de true pour matcher int/string (selon stockage)
                continue;
            }
        }
        // ========================================


        /* =====================================================
           🔵 NE COMPTER QUE type_de_contact = "Client"
           ===================================================== */
        $type = get_field('type_de_contact', $client_id);
        if (empty($type) || !in_array('Client', (array)$type, true)) {
            continue;
        }
        /* ===================================================== */


        /* =====================================================
           🔵 FIX demandé:
           - seulement pour le mode "all" (filter_user=false)
           - exclure des totaux les clients non_assigné / sans_effet
           ===================================================== */
        if ($filter_user === false) {
            if (ah_is_special_conseiller_client2($client_id)) {
                continue;
            }
        }
        /* ===================================================== */


        // Si on arrive ici : client valide → on peut l’ajouter au total
        $total_clients++;

        // === Vérification conformité ===
        $found = 0;

        foreach ($repeater_map as $repeater => $sub_field) {
            $rows = get_field($repeater, $client_id);

            if (!empty($rows) && is_array($rows)) {
                foreach ($rows as $row) {
                    $v = $row[$sub_field] ?? null;
                    $is_valid = false;

                    if (is_array($v)) {
                        if (!empty($v['ID']) || !empty($v['url']) || !empty($v['filename'])) {
                            $is_valid = true;
                        } else {
                            foreach ($v as $vv) {
                                if (is_array($vv) && !empty($vv['ID'])) { $is_valid = true; break; }
                                if (!is_array($vv) && !empty($vv)) { $is_valid = true; break; }
                            }
                        }
                    } else {
                        if (!empty($v)) $is_valid = true;
                    }

                    if ($is_valid) {
                        $found++;
                        break;
                    }
                }
            }
        }

        // --- ARRONDI CORRIGÉ ---
        $pourcentage = ($total_rep > 0) ? (int) round(($found / $total_rep) * 100) : 0;

        if ($pourcentage === 100) {
            $clients_conformes++;
        }
    }

    return [$clients_conformes, $total_clients];
}


/* ============================================================
   SHORTCODE 1 : Clients conformes du conseiller connecté
   (PAS d'exclusion non_assigné/sans_effet ici)
   ============================================================ */
add_shortcode('clients_conformes', function() {
    list($ok, $total) = acf_count_clients_conformes(true);
    return "{$ok} CLIENTS CONFORMES SUR {$total}";
});


/* ============================================================
   SHORTCODE 2 : Clients conformes (tous les clients)
   (EXCLUT non_assigné/sans_effet)
   ============================================================ */
add_shortcode('clients_conformes_all', function() {
    list($ok, $total) = acf_count_clients_conformes(false);
    return "{$ok} CLIENTS CONFORMES SUR {$total}";
});