<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms

namespace DynamicShortcodes\Core;

use DSHDeps\DynamicOOO\PluginUtils\LicenseInterface;
class LicenseFacade implements LicenseInterface {

	const DCE_LICENSE = 'dce';
	const DSH_LICENSE = 'dsh';
	const DTB_LICENSE = 'dtb';

	/**
	 * @var \DSHDeps\DynamicOOO\PluginUtils\License
	 */
	public $dsh_license;

	/**
	 * @var \DSHDeps\DynamicOOO\PluginUtils\License
	 */
	private $dce_license_system = false;

	/**
	 * @var \DSHDeps\DynamicOOO\PluginUtils\License
	 */
	private $dtb_license_system = false;

	/**
	 * @var \DSHDeps\DynamicOOO\PluginUtils\LicenseInterface
	 */
	private $used_license;

	/**
	 * @param \DSHDeps\DynamicOOO\PluginUtils\License $license
	 */
	public function __construct( $license ) {
		$this->dsh_license  = $license;
		$this->used_license = $this->dsh_license;
		add_action( 'dynamic-content-for-elementor/elementor-init', [ $this, 'dce_init' ] );
		add_action( 'dynamic-toolbox-for-bricks/init', [ $this, 'dtb_init' ] );
		add_action( 'admin_notices', [ $this, 'activation_advisor' ] );
	}

	/**
	 * @return void
	 */
	public function init() {}

	/**
	 * @return void
	 */
	public function activation_advisor() {
		if ( ! $this->is_license_active() ) {
			$this->dsh_license->activation_advisor();
		}
	}

	/**
	 * @return string
	 */
	public function get_plugin() {
		return $this->used_license->get_plugin();
	}

	/**
	 * @return boolean
	 */
	public function is_dce_installed() {
		return $this->dce_license_system !== false;
	}

	/**
	 * @return boolean
	 */
	public function is_dtb_installed() {
		return $this->dtb_license_system !== false;
	}

	/**
	 * @param \DynamicContentForElementor\Plugin $dce_plugin
	 * @return void
	 */
	public function dce_init( $dce_plugin ) {
		$this->dce_license_system = $dce_plugin->license_system;
		if ( ! $this->dsh_license->is_license_active()
			&& $this->dce_license_system->is_license_active() ) {
			$this->used_license = $this->dce_license_system;
		}
	}

	/**
	 * @param \DynamicContentForElementor\Plugin $dce_plugin
	 * @return void
	 */
	public function dtb_init( $plugin ) {
		$this->dtb_license_system = $plugin->plugin_utils->license;
		if ( ! $this->dsh_license->is_license_active()
			&& $this->dtb_license_system->is_license_active() ) {
			$this->used_license = $this->dtb_license_system;
		}
	}

	/**
	 * @return string
	 */
	public function get_license_key() {
		return $this->used_license->get_license_key();
	}

	/**
	 * @return string
	 */
	public function get_license_error() {
		return $this->used_license->get_license_error();
	}

	/**
	 * @return string
	 */
	public function get_license_key_last_4_digits() {
		return $this->used_license->get_license_key_last_4_digits();
	}

	/**
	 * @return string
	 */
	public function get_last_active_domain() {
		return $this->used_license->get_last_active_domain();
	}

	/**
	 * @param string $domain
	 * @return void
	 */
	public function set_last_active_domain( $domain ) {
		return $this->used_license->set_last_active_domain( $domain );
	}

	/**
	 * @return string
	 */
	public function get_current_domain() {
		return $this->used_license->get_current_domain();
	}

	/**
	 * @return void
	 */
	public function refresh_license_status() {
		if ( $this->used_license->get_plugin() === DYNAMIC_SHORTCODES_PREFIX ) {
			return $this->dsh_license->refresh_license_status();
		}
	}

	/**
	 * @return void
	 */
	public function refresh_and_repair_license_status() {
		if ( $this->used_license->get_plugin() === DYNAMIC_SHORTCODES_PREFIX ) {
			return $this->dsh_license->refresh_and_repair_license_status();
		}
	}

	/**
	 * @return array{0:bool,1:string}
	 */
	public function deactivate_license() {
		if ( $this->used_license->get_plugin() === DYNAMIC_SHORTCODES_PREFIX ) {
			return $this->dsh_license->deactivate_license();
		}
		return [ true, '' ];
	}

	/**
	 * @param string $key
	 * @return array{0:bool,1:string}
	 */
	public function activate_new_license_key( $key ) {
		return $this->dsh_license->activate_new_license_key( $key );
	}

	public function activate_beta_releases() {
		return $this->dsh_license->activate_beta_releases();
	}

	public function deactivate_beta_releases() {
		return $this->dsh_license->deactivate_beta_releases();
	}

	/**
	 * @param boolean $fresh
	 * @return boolean
	 */
	public function is_license_active( $fresh = false ) {
		return $this->used_license->is_license_active( $fresh );
	}

	/**
	 * @return void
	 */
	public function domain_mismatch_check() {
		return $this->used_license->domain_mismatch_check();
	}

	/**
	 * Determine which license is currently active
	 *
	 * @return string|false
	 */
	public function get_active_license() {
		if ( $this->used_license === $this->dsh_license && $this->used_license->is_license_active() ) {
			return self::DSH_LICENSE;
		} elseif ( $this->used_license === $this->dce_license_system && $this->used_license->is_license_active() ) {
			return self::DCE_LICENSE;
		} elseif ( $this->used_license === $this->dtb_license_system && $this->used_license->is_license_active() ) {
			return self::DTB_LICENSE;
		}
		return false;
	}
}
