<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms
namespace DynamicShortcodes\Core\Settings\AdminPages;

use DynamicShortcodes\Plugin;
use DSHDeps\DynamicOOO\PluginUtils\AdminPages\Pages\LicensePage;

class License extends LicensePage {
	/**
	 * @return void
	 */
	public function render_content() {
		$license_facade = Plugin::instance()->license_facade;
		$dsh_license    = $license_facade->dsh_license;
		$active_license = $license_facade->get_active_license();
		$dce_installed  = $license_facade->is_dce_installed();
		$dtb_installed  = $license_facade->is_dtb_installed();

		if ( $active_license === $license_facade::DSH_LICENSE || ( ! $dce_installed && ! $dtb_installed ) ) {
			parent::render_content();
			return;
		}

		$notices = Plugin::instance()->settings_manager->admin_pages->notices;
		?>
		<div class="wrap">
			<?php
			if ( 'POST' === $_SERVER['REQUEST_METHOD'] &&
				( ! isset( $_POST[ $this->plugin_utils_manager->get_config( 'prefix' ) . '-settings-page' ] ) ||
				! wp_verify_nonce(
					$_POST[ $this->plugin_utils_manager->get_config( 'prefix' ) . '-settings-page' ],
					$this->plugin_utils_manager->get_config( 'prefix' ) . '-settings-page'
				) ) ) {
				wp_die( esc_html__( 'Nonce verification error.', 'dynamic-shortcodes' ) );
			}

			if ( isset( $_POST['license_key'] ) ) {
				if ( $_POST['license_activated'] ) {
					list( $success, $msg ) = $dsh_license->deactivate_license();
					if ( ! $success ) {
						$notices->error( $msg );
					} else {
						$msg = esc_html__( 'License key successfully deactivated for this site', 'dynamic-shortcodes' );
						$notices->success( $msg );
					}
				} else {
					$license_key           = sanitize_text_field( $_POST['license_key'] );
					list( $success, $msg ) = $dsh_license->activate_new_license_key( $license_key );
					Plugin::instance()->plugin_utils->update_checker->clear_update_cache();
					if ( ! $success ) {
						$notices->error( $msg );
					} else {
						$msg = esc_html__( 'License key successfully activated for this site', 'dynamic-shortcodes' );
						$notices->success( $msg );
					}
				}
			} else {
				$dsh_license->refresh_and_repair_license_status();
			}
			if ( $dsh_license->domain_mismatch_check() ) {
				$notices->warning(
					esc_html__( 'License Mismatch. Your license key doesn\'t match your current domain. This is likely due to a change in the domain URL. You can reactivate your license now. Remember to deactivate the one for the old domain from your license area on Dynamic.ooo\'s site', 'dynamic-content-for-elementor' )
				);
			}

			$license_key       = $dsh_license->get_license_key();
			$is_license_active = $license_facade->is_license_active();

			?>
			<div class="content-wrapper">
				<div class="dynamic-shortcodes-tab">
					<?php $this->render_licenses_tabs(); ?>

					<?php
					if ( ! $is_license_active ) {
						$this->render_get_it_now();
					}
					?>
				</div>

				<?php
				if ( $is_license_active ) {
					$this->maybe_render_rollback();
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * @return void
	 */
	protected function render_licenses_tabs() {
		$license_facade = Plugin::instance()->license_facade;
		$dsh_license    = $license_facade->dsh_license;
		$active_license = $license_facade->get_active_license();
		$dsh_key        = $dsh_license->get_license_key();
		$dce_installed  = $license_facade->is_dce_installed();
		$dtb_installed  = $license_facade->is_dtb_installed();

		echo '<h2 class="dsh-license-status">';
		esc_html_e( 'License Status', 'dynamic-shortcodes' );
		echo ': ';

		if ( $active_license === $license_facade::DSH_LICENSE ) {
			echo "<span class='dsh-license-status-active'>";
			esc_html_e( 'Active', 'dynamic-shortcodes' );
			echo '</span>';
		} elseif ( $active_license === $license_facade::DCE_LICENSE ) {
			echo "<span class='dsh-license-status-active'>";
			esc_html_e( 'Active via Dynamic Content for Elementor', 'dynamic-shortcodes' );
			echo '</span>';
		} elseif ( $active_license === $license_facade::DTB_LICENSE ) {
			echo "<span class='dsh-license-status-active'>";
			esc_html_e( 'Active via Dynamic Toolbox for Bricks', 'dynamic-shortcodes' );
			echo '</span>';
		} else {
			echo "<span class='dsh-license-status-inactive'>";
			esc_html_e( 'Not Active', 'dynamic-shortcodes' );
			echo '</span>';
		}

		echo '</h2>';

		echo '<div class="dsh-content-wrapper">';

		// Dynamic Shortcodes license tab
		echo '<div class="dsh-license-tab">';
		echo '<h3>' . esc_html__( 'Activation with Dynamic Shortcodes License Key', 'dynamic-shortcodes' ) . '</h3>';
		$this->render_form();
		echo '<p>' . esc_html__( 'Allows full plugin functionality.', 'dynamic-shortcodes' ) . ' ' . esc_html__( 'Works with:', 'dynamic-shortcodes' ) . '<br />';
		?>
		<ul>
			<li>Dynamic Content for Elementor</li>
			<li>Elementor Free</li>
			<li>Elementor Pro</li>
			<li>Dynamic Toolbox for Bricks</li>
			<li>Bricks</li>
			<li>WPBakery</li>
			<li>Classic Editor</li>
			<li>Gutenberg (Beta)</li>
			<li>Full Site Editing (Beta)</li>
			<li>Oxygen</li>
			<li>Breakdance</li>
		</ul>
		<?php
		echo '</p>';
		echo '</div>';

		if ( $dce_installed ) {
			// Dynamic Content for Elementor tab - shown only if DCE is installed
			echo '<div class="dsh-license-tab">';
			echo '<h3>' . esc_html__( 'Activation via Dynamic Content for Elementor', 'dynamic-shortcodes' ) . '</h3>';
			echo '<a href="' . esc_url( admin_url( 'admin.php?page=dce-license' ) ) . '" class="button button-primary">' . esc_html__( 'Go to Dynamic Content for Elementor License Page', 'dynamic-shortcodes' ) . '</a>';

			if ( $active_license === $license_facade::DCE_LICENSE ) {
				?>
				<p><strong class='dsh-license-status-active'>
				<?php esc_html_e( 'Activated', 'dynamic-shortcodes' ); ?>
				</strong></p>
				<?php
			} else {
				?>
				<p><strong class='dsh-license-status-inactive'>
				<?php esc_html_e( 'Deactivated', 'dynamic-shortcodes' ); ?>
				</strong></p>
				<?php
			}

			echo '<p>' . esc_html__( 'Works with:', 'dynamic-shortcodes' ) . '<br />';
			?>
			<ul>
				<li>Dynamic Content for Elementor</li>
				<li>Elementor Free</li>
				<li>Elementor Pro</li>
			</ul>
			<?php
			echo '</div>';
		} elseif ( $dtb_installed ) {
			// Dynamic Toolbox for Bricks tab - shown only if DTB is installed
			echo '<div class="dsh-license-tab">';
			echo '<h3>' . esc_html__( 'Activation via Dynamic Toolbox for Bricks', 'dynamic-shortcodes' ) . '</h3>';
			echo '<a href="' . esc_url( admin_url( 'admin.php?page=dtb-license' ) ) . '" class="button button-primary">' . esc_html__( 'Go to Dynamic Toolbox for Bricks License Page', 'dynamic-shortcodes' ) . '</a>';

			if ( $active_license === $license_facade::DTB_LICENSE ) {
				?>
				<p><strong class='dsh-license-status-active'>
				<?php esc_html_e( 'Activated', 'dynamic-shortcodes' ); ?>
				</strong></p>
				<?php
			} else {
				?>
				<p><strong class='dsh-license-status-inactive'>
				<?php esc_html_e( 'Deactivated', 'dynamic-shortcodes' ); ?>
				</strong></p>
				<?php
			}

			echo '<p>' . esc_html__( 'Works with:', 'dynamic-shortcodes' ) . '<br />';
			?>
			<ul>
				<li>Dynamic Toolbox for Bricks</li>
				<li>Bricks</li>
			</ul>
			<?php
			echo '</div>';
		}

		echo '</div>';
	}

	/**
	 * @return void
	 */
	protected function render_form() {
		$license_facade    = Plugin::instance()->license_facade;
		$dsh_license       = $license_facade->dsh_license;
		$is_license_active = $dsh_license->is_license_active();
		$license_key       = $dsh_license->get_license_key();
		$license_domain    = get_option( DYNAMIC_SHORTCODES_PREFIX . '_license_domain' );

		?>
		<form action="" method="post" class="dsh-license-key">
			<?php wp_nonce_field( $this->plugin_utils_manager->get_config( 'prefix' ) . '-settings-page', $this->plugin_utils_manager->get_config( 'prefix' ) . '-settings-page' ); ?>
			<input type="password" autocomplete="off" name="license_key" value="<?php echo esc_attr( $license_key ); ?>" placeholder="<?php echo esc_attr_e( 'Insert Dynamic Shortcodes License Key', 'dynamic-shortcodes' ); ?>" id="license_key">
			<input type="hidden" name="license_activated" value="<?php echo esc_attr( $is_license_active ); ?>">
			<?php
			if ( $is_license_active ) {
				submit_button( esc_html__( 'Deactivate', 'dynamic-shortcodes' ), 'cancel', 'submit', false );
			} else {
				submit_button( esc_html__( 'Save key and activate', 'dynamic-shortcodes' ), 'primary', 'submit', false );
			}
			?>
		</form>

		<?php
		if ( $is_license_active ) {
			if ( $license_domain && $license_domain !== $dsh_license->get_current_domain() ) {
				?>
				<p><strong style="color:#f0ad4e;"><?php esc_html_e( 'Your license is valid but there is something wrong: license mismatch.', 'dynamic-shortcodes' ); ?></strong></p>
				<p><?php esc_html_e( 'Your license key doesn\'t match your current domain. This is most likely due to a change in the domain URL. Please deactivate the license and reactivate it', 'dynamic-shortcodes' ); ?></p>
				<?php
			} else {
				?>
				<p><strong class='dsh-license-status-active'>
					<?php
					printf(
						// translators: part of license key
						esc_html__( 'Your license ending in "%1$s" is valid and active.', 'dynamic-shortcodes' ),
						'<strong>' . esc_html( $dsh_license->get_license_key_last_4_digits() ) . '</strong>'
					);
					?>
				</strong></p>
				<?php
			}
		} else {
			?>
			<p><strong class='dsh-license-status-inactive'>
			<?php esc_html_e( 'Deactivated', 'dynamic-shortcodes' ); ?>
			</strong></p>
			<?php
		}
	}

	/**
	 * @return void
	 */
	protected function render_get_it_now() {
		echo '<div class="dsh-license-get-it-now">';
		echo '<p>' . esc_html__( 'Enter your license to keep the plugin updated, obtaining new features, future compatibility, increased stability, security, and technical support.', 'dynamic-content-for-elementor' ) . '</p>';
		//phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '<p>' . esc_html__( 'You still don\'t have one?', 'dynamic-content-for-elementor' ) . ' <a href="' . DYNAMIC_SHORTCODES_PRICING_URL . '" class="button button-small" target="_blank">' . esc_html__( 'Get it now!', 'dynamic-content-for-elementor' ) . '</a></p>';
		echo '</div>';
	}
}
