<?php

// SPDX-FileCopyrightText: 2022-2025 Ovation S.r.l. <help@dynamic.ooo>
// SPDX-License-Identifier: LicenseRef-GPL-3.0-with-dynamicooo-additional-terms
namespace DynamicShortcodes\Core\Settings\AdminPages;

use DynamicShortcodes\Plugin;
use DSHDeps\DynamicOOO\PluginUtils\AdminPages\Pages\Base;
use DynamicShortcodes\Core\LicenseFacade;
use DynamicShortcodes\Core\Settings\Manager;
use DynamicShortcodes\Core\Settings\AdminPages\Utilities;

class Settings extends Base {
	/**
	 * @return void
	 */
	public function init() {
		add_action( 'admin_init', [ $this, 'register_settings' ] );
		if ( get_option( Manager::OPTIONS_PREFIX . 'enabled_in_post_title' ) === false ) {
			add_action( 'init', [ $this, 'initialize_settings_with_activation' ], 9999 );
		}
	}

	public function initialize_settings_with_activation() {
		$license = Plugin::instance()->license_facade->get_active_license();
		if ( ! $license ) {
			return;
		}
		$options = $this->get_setting_options();
		foreach ( $options as $option ) {
			$active = $option['default'];
			if (
				$license !== 'dsh' &&
				( $option['with-license'] ?? '' ) !== $license
			) {
				$active = false;
			}

			update_option( Manager::OPTIONS_PREFIX . $option['id'], $active );
		}
	}

	/**
	 * @return void
	 */
	public function render_content() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$messages_id = Manager::MAIN_PAGE_ID . '_messages';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['settings-updated'] ) ) {
			add_settings_error( $messages_id, $messages_id, esc_html__( 'Settings Saved', 'dynamic-shortcodes' ), 'updated' );
		}
		$this->maybe_render_license_notice( $messages_id );
		// show error/update messages
		settings_errors( $messages_id );
		echo '<div class=wrap>';
		echo '<form action=options.php method=post>';
		settings_fields( Manager::MAIN_PAGE_ID );
		echo '<table class=form-table>';
		do_settings_fields( Manager::MAIN_PAGE_ID, 'dynamic_shortcodes_general' );
		echo '</table>';
		submit_button( esc_html__( 'Save Settings', 'dynamic-shortcodes' ) );
		echo '</form>';
		echo '</div>';
	}

	/**
	 * @return void
	 */
	protected function maybe_render_license_notice( $messages_id ) {
		$license_facade    = Plugin::instance()->license_facade;
		$plugin_in_use     = $license_facade->get_plugin();
		$is_license_active = $license_facade->is_license_active();

		if ( $plugin_in_use === DYNAMIC_SHORTCODES_PREFIX && $is_license_active ) {
			return;
		}

		if ( ! $is_license_active ) {
			// translators: 1 and 2 are html tags.
			$notice = esc_html__(
				'You are not able to change the settings because the license is not active. Please %1$sactivate the license%2$s.',
				'dynamic-shortcodes'
			);
		} else {
			// translators: 1 and 2 are html tags, 3 is the plugin name.
			$notice = esc_html__(
				'Your license is provided by %3$s. Contrary to the normal license not all settings can be changed. Check the %1$slicense page%2$s for more information.',
				'dynamic-shortcodes'
			);
		}

		$license_url = esc_attr( admin_url( '/admin.php?page=' . Plugin::instance()->plugin_utils->get_license_admin_page() ) );

		$notice = sprintf(
			$notice,
			"<a href=\"{$license_url}\">",
			'</a>',
			esc_html( Utilities::get_plugin_name( $plugin_in_use ) )
		);

		add_settings_error( $messages_id, $messages_id, $notice, 'warning' );
	}

	/**
	 * @return void
	 */
	protected function register_setting( $page_id, $prefix, $field ) {
		if ( ! isset( $field['sanitize_callback'] ) ) {
			register_setting( Manager::MAIN_PAGE_ID, $prefix . $field['id'] );
		} else {
			register_setting( Manager::MAIN_PAGE_ID, $prefix . $field['id'], [ 'sanitize_callback' => $field['sanitize_callback'] ] );
		}
	}

	/**
	 * @return void
	 */
	public function register_settings() {
		foreach ( $this->get_setting_fields() as $field ) {
			$subfields = $field['subfields'] ?? false;
			$prefix    = Manager::OPTIONS_PREFIX;
			if ( ! $subfields ) {
				$this->register_setting( Manager::MAIN_PAGE_ID, $prefix, $field );
			} else {
				foreach ( $subfields as $subfield ) {
					$this->register_setting( Manager::MAIN_PAGE_ID, $prefix, $subfield );
				}
			}
			add_settings_field(
				$prefix . $field['id'],
				$field['label'],
				function () use ( $field ) {
					$this->render_field( $field );
				},
				Manager::MAIN_PAGE_ID,
				'dynamic_shortcodes_general'
			);
		}
	}

	/**
	 * @return void
	 */
	protected function get_setting_fields() {
		$fields = [
			[
				'id' => 'locations',
				'label' => esc_html__( 'Where to enable Dynamic Shortcodes', 'dynamic-shortcodes' ),
				'subfields' => $this->get_setting_options(),
			],
			[
				'id' => 'more',
				'label' => esc_html__( 'More Settings', 'dynamic-shortcodes' ),
				'subfields' => [
					[
						'id' => 'hide_errors_in_frontend',
						'type' => 'checkbox',
						'label' => esc_html__( 'Hide errors and warnings in frontend', 'dynamic-shortcodes' ),
						'default' => false,
						'no-license-checks' => true,
					],
				],
			],
		];
		return $fields;
	}

	/**
	 * @return void
	 */
	protected function render_field( $field ) {
		$subfields = $field['subfields'] ?? false;
		if ( $subfields ) {
			foreach ( $subfields as $field ) {
				$this->render_field( $field );
			}
			return;
		}
		$license_facade = Plugin::instance()->license_facade;
		$active_license = $license_facade->get_active_license();

		$readonly = false;
		if ( ! isset( $field['no-license-checks'] ) ) {
			$readonly = false;

			// No license active
			if ( ! $active_license ) {
				$readonly = true;
			}
			// We are using DSH license and it's active
			elseif ( $active_license === LicenseFacade::DSH_LICENSE ) {
				$readonly = false;
			} else {
				$readonly = $active_license !== ( $field['with-license'] ?? '' );
			}
		}
		switch ( $field['type'] ) {
			case 'checkbox':
				$field_id = Manager::OPTIONS_PREFIX . $field['id'];
				$value    = get_option( $field_id );
				echo '<label>';
				$checked  = checked( $value, true, false );
				$disabled = $readonly ? 'disabled' : '';
				// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
				echo "<input type=checkbox value=1 name='$field_id' $checked $disabled>";
				echo $field['label'];
				echo '</label><br>';
				// phpcs:enable
				break;
			case 'custom':
				$field['render_callback']();
				break;
		}
	}

	/**
	 * 'default' refers to what happens when settings are changed on
	 *  activation, otherwise options are always disabled by default.
	 */
	protected function get_setting_options() {
		$bricks_label = esc_html__( 'Bricks', 'dynamic-shortcodes' );
		if ( defined( 'BRICKS_VERSION' ) ) {
			$version = preg_replace( '/-(beta|dev|rc)\d*$/', '', BRICKS_VERSION );
			if ( version_compare( $version, '1.9.9', '<' ) ) {
				$bricks_label .= esc_html__( ' (your Bricks version is not supported! Bricks 1.9.9 or later is required) ', 'dynamic-shortcodes' );
			}
		}
		return [
			[
				'id' => 'enabled_in_post_title',
				'type' => 'checkbox',
				'label' => esc_html__( 'Post Title', 'dynamic-shortcodes' ),
				'default' => false,
			],
			[
				'id' => 'enabled_in_post_content',
				'type' => 'checkbox',
				'label' => esc_html__( 'Post Content', 'dynamic-shortcodes' ),
				'default' => false,
			],
			[
				'id' => 'enabled_in_post_excerpt',
				'type' => 'checkbox',
				'label' => esc_html__( 'Post Excerpt', 'dynamic-shortcodes' ),
				'default' => false,
			],
			[
				'id' => 'enabled_in_blocks_content',
				'type' => 'checkbox',
				'label' => esc_html__( 'Gutenberg - Blocks Content', 'dynamic-shortcodes' ),
				'default' => true,
			],
			[
				'id' => 'enabled_in_elementor_dynamic_tag',
				'type' => 'checkbox',
				'label' => esc_html__( 'Elementor - Dynamic Tag', 'dynamic-shortcodes' ),
				'default' => true,
				'with-license' => 'dce',
			],
			[
				'id' => 'enabled_in_breakdance',
				'type' => 'checkbox',
				'label' => esc_html__( 'Breakdance', 'dynamic-shortcodes' ),
				'default' => true,
			],
			[
				'id' => 'enabled_in_oxygen',
				'type' => 'checkbox',
				'label' => esc_html__( 'Oxygen', 'dynamic-shortcodes' ),
				'default' => true,
			],
			[
				'id' => 'enabled_in_wpbakery',
				'type' => 'checkbox',
				'label' => esc_html__( 'WPBakery', 'dynamic-shortcodes' ),
				'default' => true,
			],
			[
				'id' => 'enabled_as_wp_shortcode',
				'type' => 'checkbox',
				'label' => esc_html__( 'WordPress Shortcode (dsh)', 'dynamic-shortcodes' ),
				'default' => true,
			],
			[
				'id' => 'enabled_in_bricks',
				'type' => 'checkbox',
				'label' => $bricks_label,
				'default' => true,
				'with-license' => 'dtb',
			],
		];
	}
}
