<?php

/**
 * Main Elementor Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Woffice_Elementor {

    /**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Woffice_Elementor The single instance of the class.
	 */
	private static $_instance = null;

    /**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Woffice_Elementor An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

    /**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		
		add_action( 'init', [ $this, 'i18n' ] );
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}

	/**
	 * Load Textdomain
	 *
	 * Load plugin localization files.
	 *
	 * Fired by `init` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function i18n() {

		load_plugin_textdomain( 'woffice' );

	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Add Plugin actions
		add_action( 'elementor/elements/categories_registered', [ $this, 'add_woffice_widget_categories' ] );
        add_action( 'elementor/widgets/widgets_registered', [ $this, 'init_widgets' ] );
		add_action( 'elementor/editor/after_enqueue_styles', [ $this,'elementor_editor_assets' ]);
		
	}

	public function elementor_editor_assets() {
		// Register styles.
		wp_register_style('wof-elementor-style', plugins_url( 'assets/admin/editor-assets/wof-ico.css', __FILE__ ));
		wp_enqueue_style('wof-elementor-style');
	}
	

    /**
	 * Create new widget category
	 *
	 * Create a new widget category for Woffice_Elementor. All future
	 * widgets from this plugin will belongs to this category.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 */
	function add_woffice_widget_categories( $elements_manager ) {

		$categories = [];
		$categories['woffice'] =
		[
			'title' => __( 'Woffice', 'woffice' ),
			'icon'  => 'fa fa-briefcase',
		];
		

		$old_categories = $elements_manager->get_categories();
		$categories = array_merge($categories, $old_categories);
	
		$set_categories = function ( $categories ) {
			$this->categories = $categories;
		};
	
		$set_categories->call( $elements_manager, $categories );
	}

	/**
	 * Init Widgets
	 *
	 * Include widget files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( __DIR__ . '/widgets/welcome.php' );
		require_once( __DIR__ . '/widgets/fun-facts.php' );
		require_once( __DIR__ . '/widgets/projects.php' );
		require_once( __DIR__ . '/widgets/time-tracking.php' );
		require_once( __DIR__ . '/widgets/tasks.php' );
		require_once( __DIR__ . '/widgets/wikis.php' );
		require_once( __DIR__ . '/widgets/latest-posts.php' );
		require_once( __DIR__ . '/widgets/map.php' );
		require_once( __DIR__ . '/widgets/birthday.php' );
		require_once( __DIR__ . '/widgets/events.php' );
		require_once( __DIR__ . '/widgets/poll.php' );

		// // Register widgets
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Welcome_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Fun_Facts_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Projects_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Time_Tracking_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Tasks_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Wikis_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Latest_Posts_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Map_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Birthday_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Events_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Woffice_Poll_Widget() );
	} 
}

Woffice_Elementor::instance();