<?php
/**
 * Woffice - Time Tracking widget.
 *
 * This widget is for Elementor users. Which helps users to build site using elementor
 *
 * @since 1.0.0
 */

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Modules\DynamicTags\Module as TagsModule;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Border;
use Elementor\Scheme_Border;
use Elementor\Utils;

class Woffice_Time_Tracking_Widget extends \Elementor\Widget_Base {

    /**
	 * Get widget name.
	 *
	 * Retrieve Woffice Time Tracking widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'woffice-time-tracking';
	}

    /**
	 * Get widget title.
	 *
	 * Retrieve Woffice Time Tracking widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Time Tracking', 'woffice' );
	}

    /**
	 * Get widget icon.
	 *
	 * Retrieve Woffice Time Tracking widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'wof-icon-time-tracking';
	}

    /**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the Woffice Time Tracking widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'woffice' ];
	}

    /**
	 * Register Woffice Time Tracking content widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the content settings for
	 * the WPJM [jobs] shortcode.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
    protected function register_controls() {

		$this->start_controls_section(
			'woffice_timetracking_section',
			[
				'label' => esc_html__( 'Content', 'woffice' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

        $this->add_control(
			'woffice_time_tracking_tag',
			[
				'label' => esc_html__( 'Title Tag', 'woffice' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3',
			]
		);

        $this->add_control(
			'woffice_time_tracking_title',
			[
				'label' => esc_html__( 'Title', 'woffice' ),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Time Tracking', 'woffice' ),
				'placeholder' => esc_html__( 'Time Tracking', 'woffice' ),
			]
		);

        $this->add_control(
			'woffice_time_tracking_content',
			[
				'label' => esc_html__( 'Description', 'woffice' ),
				'type' => \Elementor\Controls_Manager::WYSIWYG,
				'default' => esc_html__( 'Handy time tracking widget for your users. You can export and clear it from the admin side.', 'woffice' ),
				'placeholder' => esc_html__( 'Handy time tracking widget for your users. You can export and clear it from the admin side.', 'woffice' ),
			]
		);

		$this->end_controls_section();

	}

    /**
     * Get the user's work history
     *
     * @param $user_id int
     * @param $nbr int
     * @return array
     */
    public function getLog($user_id = 0, $nbr = 10){
        if (!$user_id)
            return array();

        $saved_log = get_user_option('woffice_time_tracking', $user_id);
        $saved_log = (!is_array($saved_log)) ? array() : $saved_log;

        $log = array();

        foreach ($saved_log as $date=>$tracks) {
        	$meta = array();
        	foreach ($tracks as $track) {
				if (!isset($track['meta']) || empty($track['meta'])) {
					continue;
				}

		        $meta[] = $track['meta'];
	        }

            $log[$date] = array(
            	'total'  => woffice_get_tracking_total($tracks),
	            'meta'   => implode(', ', $meta),
	            'tracks' => $tracks
            );
        }

        // We returns the last $nbr entries
        $log_returned = array_slice($log, 0, $nbr);

        /**
         * Filter `woffice_time_tracking_log`
         *
         * @param $log_returned array - The user's log once sliced
         * @param $full_log array - The user's full log
         * @param $user_id int - The user's ID
         */
        return apply_filters('woffice_time_tracking_log', $log_returned, $saved_log, $user_id);
    }

    /**
	 * Render [Job] output from WP Job Manager on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $title = $settings['woffice_time_tracking_title'];
        $class = (woffice_tracking_is_working()) ? 'is-tracking' : '';
        $log = $this->getLog(get_current_user_id());
?>
        <div class="woffice-time-tracking <?php echo $class; ?>">
            <div class="box">
                <?php if(!is_user_logged_in()) { ?>

                    <div class="woffice-time-tracking-head">
                        <i class="fa fa-lock"></i>
                        <div class="intern-box box-title">
                            <h3><?php _e('Sorry ! It is only for logged users.','woffice'); ?></h3>
                        </div>
                    </div>

                <?php } ?>

                <div class="woffice-time-tracking-head">
                    <!-- <i class="fa fa-clock"></i> -->
                    <?php if(!empty($title)) : ?>
                        <div class="intern-box box-title">
                            <?php
                                echo sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['woffice_time_tracking_tag'] ), $this->get_render_attribute_string( 'woffice_time_tracking_title' ), $title );
                            ?>
                        </div>
                    <?php endif; ?>
                </div>

                <div class="woffice-time-tracking-content">
                    <div class="woffice-time-tracking-view">
                        <?php if( isset($settings['woffice_time_tracking_content']) && !empty($settings['woffice_time_tracking_content'])) { ?>
                            <p><?php echo $settings['woffice_time_tracking_content']; ?></p>
                        <?php } ?>
                        <div class="woffice-time-tracking_time-displayed"><?php echo woffice_current_tracking_time(); ?></div>
                    </div>
                    <div class="woffice-time-tracking-view text-left" style="display: none;">
                        <?php if(!empty($log)) : ?>
                            <?php foreach ($log as $day=>$entry) : ?>
                                <div class="woffice-time-tracking-day"
                                    data-toggle="popover"
                                    data-placement="top"
                                    data-content="<?php echo esc_html($entry['meta']); ?>">
                                    <span class="badge badge-primary badge-pill" data-tracking-day="<?php echo $day; ?>"><?php echo $day; ?></span>
                                    <span data-tracking-hours="<?php echo $entry['total']; ?>">
                                        <?php echo $entry['total']; ?>
                                        <?php if ($entry['meta']) : ?>
                                            <i class="fa fa-external-link-alt text-light"></i>
                                        <?php endif; ?>
                                    </span>
                                </div>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <p><?php _e('No tracks so far. Get started!', 'woffice'); ?></p>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="woffice-time-tracking-actions">
                    <a href="#" class="woffice-time-tracking-history-toggle btn btn-default btn-sm">
                        <i class="fa-history fa"></i> <?php _e('Tracks', 'woffice'); ?>
                    </a>
                    <a href="#"
                    data-action="modal"
                    class="woffice-time-tracking-state-toggle btn btn-default btn-info btn-sm start <?php echo esc_attr(woffice_tracking_is_working() ? "d-none" : "")?>">
                        <i class="fa fa-play"></i> <span><?php _e('Start', 'woffice'); ?></span>
                    </a>

                    <a href="#"
                    data-action="stop"
                    class="woffice-time-tracking-state-toggle btn btn-default btn-danger btn-sm stop <?php echo esc_attr(woffice_tracking_is_working() ? "" : "d-none")?>">
                        <i class="fa fa-stop"></i> <span><?php _e('Stop', 'woffice'); ?></span>
                    </a>
                </div>

            </div>
        </div>

<?php
	}
}